/*

ReceiveLocation.cs. 

Copyright 2004 Brian Loesgen, Neudesic LLC.

You may use this code royalty-free, re-distribute it, modify it or deploy it at client sites, 
provided that this header information is included and remains intact.

*/

using System;
using System.Reflection;
using Microsoft.BizTalk.ExplorerOM;


namespace Neudesic.BTSUtilities
{
	/// <summary>
	/// There is one ReceiveLocations instance for each receive location that
	/// is managed by BTSServiceWindows.
	/// </summary>
	public class ReceiveLocation
	{
		public string Name;									// name of this receive location
		public string Port;									// name of the port this location belongs to
		public double Interval;								// how often location gets enabled
		public double Duration;								// how long location stays enabled
		private System.Timers.Timer Timer; 					// Timer that controls state change
		public BtsCatalogExplorer BTSConfig;				// reference to BTS Explorer OM instance	

		private object EnableConditionInstance;				// instance of object that will be called to see if receive location can be enabled
		private MethodInfo EnableConditionMethod;			// MethodInfo for enable condition method invocation
		private bool CheckEnableCondition;					// flag: check enable condition?

		private object DisableConditionInstance;			// instance of object that will be called to see if receive location can be diabled
		private MethodInfo DisableConditionMethod;			// MethodInfo for disable condition method invocation
		private bool CheckDisableCondition;					// flag: check disable condition?

		System.Timers.ElapsedEventHandler EnableHandler;	// event handler called to enable location
		System.Timers.ElapsedEventHandler DisableHandler;	// event handler called to disable location


		public static bool LoggingEnabled;					// flag:enable logging?
		
		/// <summary>
		/// Create an instance of the ReceiveLocation class. This class interacts directly with the BizTalk
		/// receive locations, toggling them on and off by using the BizTalk Explorer Object Model.
		/// 
		/// You would not normally interact with this class, you interactions would be with the 
		/// ServiceWindowManager class that controls the ReceiveLocation instances
		/// </summary>
		public ReceiveLocation(string port, string rloc, double interval, double duration)
		{
			Timer = new System.Timers.Timer();
			EnableHandler = new System.Timers.ElapsedEventHandler(EnableLocation);
			DisableHandler = new System.Timers.ElapsedEventHandler(DisableLocation);
			Timer.Elapsed += EnableHandler;
			Timer.Enabled = false;
			Timer.Interval = interval;
			this.Port = port;
			this.Name = rloc;
			this.Interval = interval;
			this.Duration = duration;
		}
		
		/// <summary>
		/// Turn on the BizTalk receive location associated with this instance of the ReceiveLocation class
		/// </summary>
		void EnableLocation(object sender, System.Timers.ElapsedEventArgs e)
		{
			// Check to see if	1) there is no enable condition and the timer fired, or,
			//					2) the timer fired, there is an enable condition, and it evaluates to true
			if (!CheckEnableCondition || (CheckEnableCondition && (bool) EnableConditionMethod.Invoke(EnableConditionInstance, new object[] {})))
			{
				Timer.Enabled = false;
				SetState(true);
				Timer.Interval = Duration;
				Timer.Elapsed -= EnableHandler;
				Timer.Elapsed += DisableHandler;
				Timer.Enabled = true;
				if (LoggingEnabled)
					Log("Enabled location");
			} 
			else if (CheckEnableCondition && LoggingEnabled)
			{
				// if logging is enabled, record the fact that we tried to activate
				Log("Enabling condition not met");
			}
		}

		/// <summary>
		/// Turn off the BizTalk receive location associated with this instance of the ReceiveLocation class
		/// </summary>
		void DisableLocation(object sender, System.Timers.ElapsedEventArgs e)
		{
			// Check to see if	1) there is no disable condition and the timer fired, or,
			//					2) the timer fired, there is a disable condition, and it evaluates to true
			if (!CheckDisableCondition || (CheckDisableCondition && (bool) DisableConditionMethod.Invoke(DisableConditionInstance, new object[] {})))
			{
				Timer.Enabled = false;
				SetState(false);
				Timer.Interval = Interval;
				Timer.Elapsed -= DisableHandler;
				Timer.Elapsed += EnableHandler;
				Timer.Enabled = true;
				if (LoggingEnabled)
					Log("Disabled location");
			} 
			else if (CheckEnableCondition && LoggingEnabled)
			{
				// if logging is enabled, record the fact that we tried to activate
				Log("Disabling condition not met");
			}

		}

		/// <summary>
		/// Create an instance of the condition class. This is cached, and the same instance will
		/// be invoked each time
		/// </summary>
		/// <param name="AssemblyName">Name of the assembly</param>
		/// <param name="ClassName">Name of the class</param>
		/// <param name="MethodName">Name of the invoked method</param>
		/// <param name="SetEnabledCondition">flag: true if the is an enableCondition, false is not</param>
		/// <returns></returns>
		public bool SetCondition(string AssemblyName, string ClassName, string MethodName, bool SetEnabledCondition)
		{
			bool success = true;
			Assembly asmb = Assembly.LoadWithPartialName(AssemblyName);
			try
			{
				Type ClassType = asmb.GetType(ClassName); 
				if (SetEnabledCondition)
				{
					EnableConditionInstance = (Object) Activator.CreateInstance(ClassType);
					EnableConditionMethod = ClassType.GetMethod(MethodName);
					CheckEnableCondition = true;
				}
				else
				{
					DisableConditionInstance = (Object) Activator.CreateInstance(ClassType);
					DisableConditionMethod = ClassType.GetMethod(MethodName);
					CheckDisableCondition = true;
				}		
			} 
			catch (Exception ex)
			{
				Log("Unable to create objects for a condition (Assembly: " + AssemblyName+ ", Class:" + ClassName + ", Method:"+ MethodName + " Error:" + ex.Message);
				success = false;
			}
			return success;
		}


		/// <summary>
		/// Writes a trace message to the event log
		/// </summary>
		/// <param name="message">Core content of message, actually message will include header/trailer text.</param>
		void Log(string message)
		{
			message = "ServiceWindows: ("+Port+":"+Name+") " + message;
			string title = message;
			if (title.Length > 200)
				title = title.Substring(0,195)+ "... more...";
			System.Diagnostics.EventLog.WriteEntry(title,message);
		}


		/// <summary>
		/// Start timer for this ReceiveLocation object
		/// </summary>
		public void Enable()
		{
			Timer.Enabled = true;
		}

		/// <summary>
		/// Stop the timer for this ReceiveLocation object
		/// </summary>
		public void Disable()
		{
			Timer.Enabled = false;
		}


		/// <summary>
		/// Set a future time to start processing this instance of the ReceiveLocation class
		/// </summary>
		/// <param name="startTime"></param>
		public void SetStartTime(string startTime)
		{
			bool enabledState = Timer.Enabled;
			Timer.Enabled = false;
			TimeSpan ts = Convert.ToDateTime(startTime) - System.DateTime.Now;
		
			if (ts.TotalMilliseconds<0)
				ts = new TimeSpan(0,0,0,0,86400000 - (int) ts.TotalMilliseconds); // time-of-day is earlier that now, can't go negative
			Timer.Interval = ts.TotalMilliseconds;
			if (LoggingEnabled)
				Log("Waiting " + ts.ToString() + " to start processing location ");
			Timer.Enabled = enabledState;
		}

		/// <summary>
		/// Set the state of a BizTalk receive location
		/// </summary>
		/// <param name="state">bool, value to set the state to</param>
		void SetState(bool state)
		{			
			try
			{
				BTSConfig.ReceivePorts[Port].ReceiveLocations[Name].Enable = state;
				BTSConfig.SaveChanges();
			} 
			catch (Exception ex)
			{
				BTSConfig.DiscardChanges();
				throw new Exception("Unable to save BTS Explore OM changes.",ex);
			}
		}

	}
}
